<?php

namespace App\Http\Controllers;

use App\Models\Product;
use App\Models\Category;
use App\Models\InventoryItem;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class ProductController extends Controller
{
    public function index()
    {
        $products = Product::all();
        return view('products.index', compact('products'));
    }

    public function show(Product $product)
    {
        return view('products.show', compact('product'));
    }

    public function category(Category $category)
    {
        $category->load('products');
        return view('products.category', compact('category'));
    }

    public function create()
    {
        return view('products.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'required|string',
            'sku' => 'required|string|unique:products,sku',
            'category_id' => 'required|exists:categories,id',
            'use_by_date' => 'required|date',
            'wholesale_price' => 'required|numeric|min:0',
            'minimum_order_quantity' => 'required|integer|min:1',
            'available_quantity' => 'required|integer|min:1',
        ]);

        $product = Product::create([
            'name' => $request->name,
            'description' => $request->description,
            'sku' => $request->sku,
            'category_id' => $request->category_id,
            'status' => 'active',
        ]);

        InventoryItem::create([
            'product_id' => $product->id,
            'use_by_date' => $request->use_by_date,
            'wholesale_price' => $request->wholesale_price,
            'minimum_order_quantity' => $request->minimum_order_quantity,
            'available_quantity' => $request->available_quantity,
        ]);

        return redirect()->route('products.index')->with('success', 'Product created successfully.');
    }

    public function search(Request $request)
    {
        $query = $request->input('query');

        $products = Product::where('name', 'LIKE', "%{$query}%")
            ->orWhere('description', 'LIKE', "%{$query}%")
            ->paginate(10);

        return view('products.index', compact('products'))->with('query', $query);
    }

    public function sellerIndex()
    {
        $sellerId = Auth::id();
        $products = Product::where('seller_id', $sellerId)->get();
        return view('products.seller.index', compact('products'));
    }
}